//==================================================================================================================================//
/* MagCalc.cpp: Calculation of the magnetic potential, vector, gradient tensor fields and and also the third-order partial          */
/*   derivatives of the magnetic potential field from spherical harmonic  coefficients.                                             */
/* Authors: Jinsong Du and Chao Chen                                                                                                */
/* Main organization: Institute of Geophysics & Geomatics, China University of Geosciences, Wuhan 430074, China                     */
/* Hisdory:                                                                                                                         */
/*   Version 0.0-31 January 2014;                                                                                                   */
/*   Version 0.1-15 April 2015: Adding the third-order partial derivatives of the magnetic potential field                          */
/* Notes:                                                                                                                           */
/*   (1) Geocentric spherical coordinate system is used to describe the locations of the computing points;                          */
/*   (2) Magnetic  fields are expressed in the spherical local north-oriented reference frame;                                      */
/*   (3) Output and units are described in the code.                                                                                */
/*   (4) This code is only available for the calculation of the internal magnetic field, for example, the expression of the magnetic*/
/*       potential is V=a*sum(sum((a/r)^(l+1)*(glm*cos(m*fai)+hlm*sin(m*fai)))*Plm(cos(theta))));                                   */
/*   (5) This code is already tested on the Windows system and can be compiled by Microsoft Visual C++ or Microsoft Visual Studio.  */                                                                                                         
/* Reference:                                                                                                                       */
/*   Jinsong Du, Chao Chen, Vincent Lesur, and Linsong Wang: Non-singular spherical harmonic expressions of geomagnetic vector and  */
/*   gradient tensor fields in the local north-oriented reference frame, Geoscientific Model Development: Discussion (under review  */
/*   for Geoscientific Model Development), 7, 8477-8503, 2014.                                                                      */
//==================================================================================================================================//
/* If you have any question, please contact with us:                                                                                */
/* E-mail: jinsongdu@cug.edu.cn, jinsongdu.cug@gmail.com, chenchao@cug.edu.cn                                                       */
//==================================================================================================================================//
// Corresponding head files
#include <stdio.h>
#include <math.h>
#include <mbstring.h>
#include <stdio.h>
#include <conio.h>
#include <time.h>
#include <memory.h>
// Circumference ratio
#ifndef PI
#define PI 3.1415926535897932384626433832795
#endif
// Schmidt semi-normalized associated Legendre function
void   Belikov(long Nmax,double theta,double** pwave,double** vp);
double Nwave(long n,long m);
// Function for converting unit of angle from degree to rad
double ANG2ARC(double x);
// Function for converting unit of angle from rad to degree
double ARC2ANG(double x);
//==================================================================================================================================//
int main()
{
	long i,j,m,n,l,k;
	time_t start,end;
	// Maximum degree
    long Nmax;
	// Incipient degree (it must be larger or equal to one)
    long N_start;
	// Terminational degree (it must be smaller or equal to Nmax)
	long N_end;
	// Geomagnetic reference sphere (i.e. 6371.2 km) (unit: m)
	double R;
	// Altitude refered to the geomagnetic reference sphere (i.e. 6371.2 km) (unit: m)
    double Altitude;
	// Griding space in latitude (unit: degree)
    double dW;
	// Griding space in longitude (unit: degree)
	double dJ;
	// Incipient latitude (unit: degree)
	double Ws;
	// Terminational latitude (unit: degree)
	double We;
	// Incipient longitude (unit: degree)
	double Js;
	// Terminational longitude (unit: degree)
	double Je;
	//=============================================================================================================================//
	// Setting the required parameters
	Nmax=120;
	N_start=16;
	N_end=90;
	R=6371.2E+003;
	Altitude=300.0E+003;
	dW=0.125;
	dJ=0.125;
	Ws=+60.0;
	We=+90.0;
	Js=-180.0;
	Je=+180.0;
	// Open the input file of the spherical harmonic  coefficients 
	FILE* fpIn=fopen("GRIMM_L120_V0.0.cof","rt");
	if(fpIn==NULL)
    {
        printf("\nSorry!\nFailed to open the input file:");
        printf( ".\n");
        return -1;
    }
	// Open the output file of the the magnetic fields
	FILE *fpOut=fopen("GRIMM_L120_V0.0_300km_NorthPole_0.125degree&0.125degree_1-90degree.dat", "w+");
	// Setting the header of the output file 
	fprintf(fpOut,"Lon[degree]\tLat[degree]\tV[mT*m]\tBx[nT]\tBy[nT]\tBz[nT]\tBxx[pT/m]\tBxy[pT/m]\tBxz[pT/m]\tByy[pT/m]\tByz[pT/m]\tBzz[pT/m]\tBxxz[aT/m^2]\tBxyz[aT/m^2]\tBxzz[aT/m^2]\tByyz[aT/m^2]\tByzz[aT/m^2]\tBzzz[aT/m^2]\n");
	// Magnetic potential: V (unit: mT*m)
	// Magnetic vector: Bx, By and Bz (unit: nT)
	// Megnetic gradient tensor: Bxx, Bxy, Bxz, Byy, Byz and Bzz (unit: pT/m)
	// Third-order partial derivatives of the magnetic potential field (unit: aT/m^2)
	//=============================================================================================================================//
	// Display the setting parameters
    printf("                 Max Dregree N = %ld\n", Nmax);
    printf("  The Incipient Degree N_start = %ld\n", N_start);
	printf("The Terminational degree N_end = %ld\n", N_end);
	printf("                        R = %.3lf m\n", R);
    printf("                 Altitude = %.3lf m\n", Altitude);
    printf("               Gridstep_W = %.3lf degree\n", dW);
	printf("               Gridstep_J = %.3lf degree\n", dJ);
	printf("                       Ws = %.3lf degree\n", Ws);
	printf("                       We = %.3lf degree\n", We);
	printf("                       Js = %.3lf degree\n", Js);
	printf("                       Je = %.3lf degree\n", Je);
	//=============================================================================================================================//
    long nNM = Nmax+1;
    double** vg    = new double*[nNM];
    double** vh    = new double*[nNM];
    double** vP    = new double*[nNM+2];
	double** Pwave = new double*[nNM+2];
    vg[0]    = new double[nNM*nNM];
    vh[0]    = new double[nNM*nNM];
    vP[0]    = new double[(nNM+2)*(nNM+2)];
	Pwave[0] = new double[(nNM+2)*(nNM+2)];
    for(i=1;i<nNM;++i)
    {
        vg[i]=vg[i-1]+nNM;
        vh[i]=vh[i-1]+nNM;		
    }
	for(i=1;i<(nNM+2);++i)
	{
		vP[i]   =vP[i-1]   +nNM+2;
		Pwave[i]=Pwave[i-1]+nNM+2;
	}
	for(i=0;i<(nNM+2);++i)
	{
		for(j=0;j<(nNM+2);++j)
		{
			vP[i][j]=0.0;
		}
	}

	// Number of computing points in latitude
    long nLat=(long)((We-Ws)/dW+1+1e-5);
	// Number of computing points in longitude
	long nLon=(long)((Je-Js)/dJ+1+1e-5);

    // Reading the spherical harmonic coefficiences of the geomagnetic field model
	printf("\nReading the spherical harmonic coefficiences is begining !\n");
    double C,S;
    while(!feof(fpIn))
    {
        fscanf(fpIn,"%ld\t%ld\t%lf\t%lf\n",&n,&m,&C,&S);
        vg[n][m]=C;
        vh[n][m]=S;
    }
    fclose(fpIn);
	printf("Reading the spherical harmonic coefficiences is finished !\n");
    //==================================================================================================================================//
    double phi0,lamda0;
    double phi,lamda;
	double r=R+Altitude;
	double rR=R/r;
	double rpow1,rpow2,rpow3,rpow4;
	double V,Bx,By,Bz,Bxx,Bxy,Bxz,Byy,Byz,Bzz,Bxxz,Bxyz,Bxzz,Byyz,Byzz,Bzzz;
	double Vtmp,Bxtmp,Bytmp,Bztmp;
	double Bxxtmp,Bxytmp,Bxztmp,Byytmp,Byztmp,Bzztmp;
	double Bxxztmp,Bxyztmp,Bxzztmp,Byyztmp,Byzztmp,Bzzztmp;
	double a1,a2,a3,a4;
	double ax,bx,ay,by,az,axx,bxx,cxx,axy,bxy,cxy,axz,bxz,ayy,byy,cyy,ayz,byz,azz,
		   axxz,bxxz,cxxz,axyz,bxyz,cxyz,axzz,bxzz,ayyz,byyz,cyyz,ayzz,byzz,azzz;
	double sinlamda,coslamda;
	long   nProcess=0;

	// Displaying the calculating processes
    printf("\nCalculating process : 1234");
	time(&start);
    for(l=0;l<nLat;l++)
    {
        nProcess=(long)((l+1)*100/nLat);
        printf("\b\b\b\b%2ld%% ",nProcess);

		// Latitude of the computing point
        phi0=Ws+l*dW; 
        phi=ANG2ARC(phi0); 
		
		Belikov(N_end+2,PI/2.0-phi,Pwave,vP);

        for(k=0;k<nLon;k++)
        {
			// Longitude of the computing point
            lamda0=k*dJ+Js;
            lamda=ANG2ARC(lamda0);

			Vtmp=Bxtmp=Bytmp=Bztmp=0.0;
			Bxxtmp=Bxytmp=Bxztmp=Byytmp=Byztmp=Bzztmp=0.0;
			Bxxztmp=Bxyztmp=Bxzztmp=Byyztmp=Byzztmp=Bzzztmp=0.0;

            for(i=N_start;i<=N_end;i++)
            {
				rpow1=pow(rR,i+1.0);
				rpow2=pow(rR,i+2.0);
				rpow3=pow(rR,i+3.0);
				rpow4=pow(rR,i+4.0);

                for(j=0;j<=i;j++)
                {
					a1=a2=a3=a4=1.0;
					if(j==0)
					{
						a1=sqrt(0.5);
						a2=sqrt(0.5);
						a3=sqrt(0.5);
						a4=sqrt(0.5);
					}
					if(j==1)
					{
						a1=sqrt(2.0);
					}
					if(j==2)
					{
						a3=sqrt(2.0);
					}
					ax=0.5*sqrt(i*1.0+j)*sqrt(i-j+1.0)*a1;
					bx=-0.5*sqrt(i+j+1.0)*sqrt(i*1.0-j)*a2;
					ay=0.5*sqrt(i*1.0+j)*sqrt(i+j-1.0)*a1;
					if((i-j)==0)
					{
						by=0.0;
					}
					else
					{
						by=0.5*sqrt(i*1.0-j)*sqrt(i-j-1.0)*a2;
					}
					az=-(i+1.0);
					axx=-0.25*sqrt(i*1.0+j)*sqrt(i+j-1.0)*sqrt(i-j+2.0)*sqrt(i-j+1.0)*a3;
					bxx=0.25*((i*1.0+j)*(i-j+1.0)+(i-j)*(i+j+1.0))+(i+1.0);
					if((i-j)==0)
					{
						cxx=0.0;
					}
					else
					{
						cxx=-0.25*sqrt(i+j+2.0)*sqrt(i+j+1.0)*sqrt(i*1.0-j)*sqrt(i-j-1.0)*a4;
					}
					axy=-0.25*sqrt(i*1.0+j)*sqrt(i-j+1.0)*sqrt(i-j+2.0)*sqrt(i-j+3.0)*a3;
					bxy=-0.5*j*sqrt(i-j+1.0)*sqrt(i+j+1.0);
					cxy=0.25*sqrt(i+j+1.0)*sqrt(i+j+2.0)*sqrt(i+j+3.0)*sqrt(i*1.0-j)*a4;
					axz=(i+2.0)*ax;
					bxz=(i+2.0)*bx;
					ayy=0.25*sqrt(i*1.0+j)*sqrt(i+j-1.0)*sqrt(i-j+1.0)*sqrt(i-j+2.0)*a3;
					byy=0.25*((i+j)*(i+j-1.0)+(i-j)*(i-j-1.0)*(j-1.0)/(j+1.0)+2.0*(i+j+2.0)*(i+j+1.0)/(j+1.0));
					if((i-j)==0)
					{
						cyy=0.0;
					}
					else
					{
						cyy=0.25*sqrt(i+j+1.0)*sqrt(i+j+2.0)*sqrt(i*1.0-j)*sqrt(i-j-1.0)*a4;
					}	
					ayz=(i+2.0)*ay;
					byz=(i+2.0)*by;
					azz=(i+2.0)*az;
					axxz=(i+3.0)*axx;
					bxxz=(i+3.0)*bxx;
					cxxz=(i+3.0)*cxx;
					axyz=(i+3.0)*axy;
					bxyz=(i+3.0)*bxy;
					cxyz=(i+3.0)*cxy;
					axzz=(i+3.0)*axz;
					bxzz=(i+3.0)*bxz;
					ayyz=(i+3.0)*ayy;
					byyz=(i+3.0)*byy;
					cyyz=(i+3.0)*cyy;
					ayzz=(i+3.0)*ayz;
					byzz=(i+3.0)*byz;
					azzz=(i+3.0)*azz;

					sinlamda=sin(j*lamda);
					coslamda=cos(j*lamda);

					// Magnetic potential
					Vtmp+=rpow1*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*vP[i][j];
					if(j==0)
					{
						// Magnetic vector
						Bxtmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-ax*vP[i][1]+bx*vP[i][j+1]);
						Bytmp+=rpow2*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(-ay*vP[i-1][1]+by*vP[i-1][j+1]);
						Bztmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*az*vP[i][j];
						// Magnetic gradient tensor
						Bxxtmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axx*vP[i][2]+bxx*vP[i][j]+cxx*vP[i][j+2]);
						Bxytmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(axy*vP[i+1][2]+bxy*vP[i+1][j]+cxy*vP[i+1][j+2]);
						Bxztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-axz*vP[i][1]+bxz*vP[i][j+1]);
						Byytmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ayy*vP[i][2]+byy*vP[i][j]+cyy*vP[i][j+2]);
						Byztmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(-ayz*vP[i-1][1]+byz*vP[i-1][j+1]);
						Bzztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azz*vP[i][j];
						// Third-order partial derivatives of the magnetic potential field
						Bxxztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axxz*vP[i][2]+bxxz*vP[i][j]+cxxz*vP[i][j+2]);
						Bxyztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(axyz*vP[i+1][2]+bxyz*vP[i+1][j]+cxyz*vP[i+1][j+2]);
						Bxzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-axzz*vP[i][1]+bxzz*vP[i][j+1]);
						Byyztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ayyz*vP[i][2]+byyz*vP[i][j]+cyyz*vP[i][j+2]);
						Byzztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(-ayzz*vP[i-1][1]+byzz*vP[i-1][j+1]);
						Bzzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azzz*vP[i][j];
					}
					else if(j==1)
					{
						// Magnetic vector
						Bxtmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ax*vP[i][j-1]+bx*vP[i][j+1]);
						Bytmp+=rpow2*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ay*vP[i-1][j-1]+by*vP[i-1][j+1]);
						Bztmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*az*vP[i][j];
						// Magnetic gradient tensor
						Bxxtmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-axx*vP[i][1]+bxx*vP[i][j]+cxx*vP[i][j+2]);
						Bxytmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(-axy*vP[i+1][1]+bxy*vP[i+1][j]+cxy*vP[i+1][j+2]);
						Bxztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axz*vP[i][j-1]+bxz*vP[i][j+1]);
						Byytmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-ayy*vP[i][1]+byy*vP[i][j]+cyy*vP[i][j+2]);
						Byztmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ayz*vP[i-1][j-1]+byz*vP[i-1][j+1]);
						Bzztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azz*vP[i][j];
						// Third-order partial derivatives of the magnetic potential field
						Bxxztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-axxz*vP[i][1]+bxxz*vP[i][j]+cxxz*vP[i][j+2]);
						Bxyztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(-axyz*vP[i+1][1]+bxyz*vP[i+1][j]+cxyz*vP[i+1][j+2]);
						Bxzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axzz*vP[i][j-1]+bxzz*vP[i][j+1]);
						Byyztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(-ayyz*vP[i][1]+byyz*vP[i][j]+cyyz*vP[i][j+2]);
						Byzztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ayzz*vP[i-1][j-1]+byzz*vP[i-1][j+1]);
						Bzzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azzz*vP[i][j];
					}
					else
					{
						// Magnetic vector
						Bxtmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ax*vP[i][j-1]+bx*vP[i][j+1]);
						Bytmp+=rpow2*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ay*vP[i-1][j-1]+by*vP[i-1][j+1]);
						Bztmp+=rpow2*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*az*vP[i][j];
						// Magnetic gradient tensor
						Bxxtmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axx*vP[i][j-2]+bxx*vP[i][j]+cxx*vP[i][j+2]);
						Bxytmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(axy*vP[i+1][j-2]+bxy*vP[i+1][j]+cxy*vP[i+1][j+2]);
						Bxztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axz*vP[i][j-1]+bxz*vP[i][j+1]);
						Byytmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ayy*vP[i][j-2]+byy*vP[i][j]+cyy*vP[i][j+2]);
						Byztmp+=rpow3*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ayz*vP[i-1][j-1]+byz*vP[i-1][j+1]);
						Bzztmp+=rpow3*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azz*vP[i][j];
						// Third-order partial derivatives of the magnetic potential field
						Bxxztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axxz*vP[i][j-2]+bxxz*vP[i][j]+cxxz*vP[i][j+2]);
						Bxyztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(axyz*vP[i+1][j-2]+bxyz*vP[i+1][j]+cxyz*vP[i+1][j+2]);
						Bxzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(axzz*vP[i][j-1]+bxzz*vP[i][j+1]);
						Byyztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*(ayyz*vP[i][j-2]+byyz*vP[i][j]+cyyz*vP[i][j+2]);
						Byzztmp+=rpow4*(vg[i][j]*sinlamda-vh[i][j]*coslamda)*(ayzz*vP[i-1][j-1]+byzz*vP[i-1][j+1]);
						Bzzztmp+=rpow4*(vg[i][j]*coslamda+vh[i][j]*sinlamda)*azzz*vP[i][j];
					}
				}
			}
			// Unit: 1mT*m=1E-003T*m
			V=R*Vtmp*1E-006;
			// Unit: 1nT=1E-009T
			Bx=Bxtmp;
			By=Bytmp;
			Bz=Bztmp;
			// Unit: 1pT/m=1E-012T/m
			Bxx=Bxxtmp/R*1E+003;
			Bxy=Bxytmp/R*1E+003;
			Bxz=Bxztmp/R*1E+003;
			Byy=Byytmp/R*1E+003;
			Byz=Byztmp/R*1E+003;
			Bzz=Bzztmp/R*1E+003;

			// Unit: 1aT/m^2=1E-018T/m^2
			Bxxz=Bxxztmp/(R*R)*1E+009;
			Bxyz=Bxyztmp/(R*R)*1E+009;
			Bxzz=Bxzztmp/(R*R)*1E+009;
			Byyz=Byyztmp/(R*R)*1E+009;
			Byzz=Byzztmp/(R*R)*1E+009;
			Bzzz=Bzzztmp/(R*R)*1E+009;

			// Output the calculated results
			fprintf(fpOut,"%.6lf\t%.6lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\t%.15lf\n", 
				lamda0,phi0,V,Bx,By,Bz,Bxx,Bxy,Bxz,Byy,Byz,Bzz,Bxxz,Bxyz,Bxzz,Byyz,Byzz,Bzzz);
        }
    }
	fclose(fpOut);
	time(&end);
	printf("\n\nCalculation is finished !");
	printf("\n\nComputing time: %.5lf seconds",difftime(end,start));
	//==================================================================================================================================//
    // Release the memory
	delete[] Pwave[0];    delete[] Pwave;
	delete[] vP[0];       delete[] vP;
    delete[] vg[0];       delete[] vg;
    delete[] vh[0];       delete[] vh;
   
	// Calculation is flinished
    printf("\n\nCalculating completed!\t\t\n\nPlease press any key to exit ...\n\n");
    getch();
    return 0;
}
//======================================================================================================================================//
// Schmidt semi-normalized associated Legendre function
void Belikov(long Nmax,double theta,double** pwave,double** vp)
{
	double temp=1.0;
	pwave[0][0]=1.0;
	vp[0][0]=1.0;

	for(long n=1;n<=Nmax;n++)
	{
		for(long m=0;m<=n;m++)
		{
			if(m>0)
			{	
				if((m+1)>=n)
				{
					pwave[n-1][n]=0.0;
					pwave[n-1][n+1]=0.0;
				}
				pwave[n][m]=cos(theta)*pwave[n-1][m]-sin(theta)*(0.25*pwave[n-1][m+1]-pwave[n-1][m-1]);
			}
			else
			{
				if(n<2)pwave[n-1][1]=0;
				pwave[n][m]=cos(theta)*pwave[n-1][0]-sin(theta)*0.5*pwave[n-1][1];	
			}
			// Gravitational field
			// temp=sqrt(2.0*n+1.0)*pwave[n][m]*Nwave(n,m);
			// Geomagnetic field
			temp=pwave[n][m]*Nwave(n,m);
			vp[n][m]=temp;
		}
	}
}
double Nwave(long n,long m)
{
    double barN=1.0;
	double temp1;
	long n2;
	long m2;
	
	if(n>=2)
	{
		for(long i=2;i<=n;i++)
		{
			n2=i*i;
			m2=m*m;
			
			if(m<i)
			{
				temp1=sqrt(1.0-1.0*m2/n2)*barN;
			}
			else
			{
				temp1=sqrt(1.0-0.5/i)*barN;
			}
			barN=temp1;
		}
	}
	else
	{
		barN=1.0;
	}
	return barN;
}
// Function for converting unit of angle from degree to rad
double ANG2ARC(double x)
{
	return x*PI/180.0;
}
// Function for converting unit of angle from rad to degree
double ARC2ANG(double x)
{
	return x*180.0/PI;
}